<?php
/*  *************************************************************************** *
 *	rc_uploader.class                                                           *
 *	                                                                            *
 *	a class to upload files via html form.                                      *
 *	                                                                            *
 *	@author 	Rudi Bieller, http://www.reducedcomplexity.com/                 *
 *	@version 	0.3                                                             *
 *	@requires	php >= 4.1.0 (>= 4.2.0 for php error codes)                     *
 *	                                                                            *
 *	thank you for using my stuff. please make sure to have a look at the        *
 *	terms of use before using this:                                             *
 *	http://www.reducedcomplexity.com/scripts/                                   *
 *	*************************************************************************** *
 */

define('UPLOADER_DEFAULT_CHMOD', 0660);

class uploader
{
	/**
	 *	absolute path to the storage folder
	 *	
	 *	@access private
	 *	@var String
	 */
	var $_abspath;
	/**
	 *	os delimiter / DIRECTORY_SEPARATOR ("\" on win32, "/" all others)
	 *	
	 *	@access private
	 *	@var string
	 */
	var $_delimiter;
	/**
	 *	maximum filesize allowed for an upload
	 *	
	 *	@access private
	 *	@var int
	 */
	var $_maxFilesize;
	/**
	 *	all occured errors stored in an array
	 *	
	 *	@access private
	 *	@var array
	 */
	var $_errors = array();
	/**
	 *	extensions that are allowed for upload
	 *	
	 *	@access private
	 *	@var array
	 */
	var $_allowedFiletypes = array();
	/**
	 *	the cleaned up filename
	 *	
	 *	@access private
	 *	@var string
	 */
	var $_filenameClean;
	/**
	 *	an array with values of $_FILES['inputfieldname']
	 *	
	 *	@access private
	 *	@var array
	 */
	var $_files = array();
	/* used for the email-reporting, new from v0.2 */
	/**
	 *	email address that will be used as From: / Reply-to:
	 *	
	 *	@access private
	 *	@var string
	 */
	var $_mailReportSender;
	/**
	 *	subject that will be used
	 *	
	 *	@access private
	 *	@var string
	 */
	var $_mailReportSubject;
	/**
	 *	string containing the report that will be sent to admin(s)
	 *	
	 *	@access private
	 *	@var string
	 */
	var $_mailReportMessage;
	/**
	 *	language of the reportmail
	 *	
	 *	@access private
	 *	@var string
	 */
	var $_mailReportLang;
	/**
	 *	recipients of the email-reporting
	 *	
	 *	@access private
	 *	@var array
	 */
	var $_mailReportRecipients = array();
	/**
	 *	list with all uploaded files (with abs. path)
	 *	
	 *	@access private
	 *	@var array
	 */
	var $_mailReportFileList = array();
	/**
	 *	the running version of php; new in 0.2.1
	 *	
	 *	@access private
	 *	@var string
	 */
	var $_phpvers;
	/**
	 *	array containing infos about successfully uploaded files; new in 0.2.2
	 *	
	 *	@access private
	 *	@var array
	 */
	var $_successList = array();
	/**
	 *	files will receive a chmod to this value
	 *	
	 *	@access private
	 *	@var int
	 */
	var $_chmod = UPLOADER_DEFAULT_CHMOD;
	
	
	/** 
	 *	constructor. automatically sets the delimiter and used php-version
	 */
	function uploader()
	{
		$this->_delimiter = DIRECTORY_SEPARATOR;
		$this->_setPhpVersion();
	}
	
	
	/******************************************************************/
	/************************* public methods *************************/
	/******************************************************************/
	
	
	/**
	 *	uploads files of $_FILES array
	 *	@access public
	 *	@param String $inputfieldname (name of the input field that gathers the files)
	 *	@param String $uploadFolder (folder that will store the uploaded file(s))
	 *	@param int $mode (overwrite mode if file exists: 0|do nothing, 1|overwrite, 2|autorename)
	 *	@return bool
	 */
	function upload($inputfieldname, $uploadFolder, $mode=0)
	{
		$this->_fillFilesArray($inputfieldname);
		
		if (!$this->_setAbspath($uploadFolder)) {
			return false;
		}
		
		// now get each file going...
		for ($i=0; $i<sizeof($this->_files[$inputfieldname]["tmp_name"]); $i++) {
			
			$current_filename = ''; // the original filename of the file uploaded
			// if everything is fine with the file, let's try to store it on the server
			if ($this->_checkFileForErrors($inputfieldname, $i, $current_filename) === true) {
				// cleaned up name
				$this->_filenameClean = $this->_stringCleaner($this->_files[$inputfieldname]["name"][$i]);
				// full path of the destination
				$target = $this->_abspath . $this->_delimiter . $this->_filenameClean;
				// depending on the mode setting, file(s) will be overwritten or not. or maybe renamed.
				switch ((int) $mode) {
				case 0: // do nothing if exists
					if ($this->_checkFileExists($this->_filenameClean)) {
						$this->_errorTrigger($current_filename, 5);
					} else {
						if (@move_uploaded_file($this->_files[$inputfieldname]["tmp_name"][$i], $target)) {
							$this->_errorTrigger($current_filename, 0);
							$this->_addToMailReportFileList($target);
							$this->_addToSuccessList($this->_files[$inputfieldname]["name"][$i], 
													$this->_filenameClean, 
													$this->_files[$inputfieldname]["size"][$i], 
													$this->_files[$inputfieldname]["type"][$i]);
							if (!@chmod($target, UPLOADER_DEFAULT_CHMOD)) {
								$this->_errorTrigger($current_filename, 12);
							}
						} else {
							$this->_errorTrigger($current_filename, 9);
						}
					}
					break;
				case 1: // overwrite if exists
					if (@move_uploaded_file($this->_files[$inputfieldname]["tmp_name"][$i], $target)) {
						$this->_errorTrigger($current_filename, 0);
						$this->_addToMailReportFileList($target);
						$this->_addToSuccessList($this->_files[$inputfieldname]["name"][$i], 
												$this->_filenameClean, 
												$this->_files[$inputfieldname]["size"][$i], 
												$this->_files[$inputfieldname]["type"][$i]);
						if (!@chmod($target, UPLOADER_DEFAULT_CHMOD)) {
							$this->_errorTrigger($current_filename, 12);
						}
					} else {
						$this->_errorTrigger($current_filename, 9);
					}
					break;
				case 2: // autorename if exists
					$newname = $this->_autoRename($this->_filenameClean);
					$target = $this->_abspath . $this->_delimiter . $newname;
					if (@move_uploaded_file($this->_files[$inputfieldname]["tmp_name"][$i], $target)) {
						$this->_errorTrigger($current_filename, 0);
						$this->_addToMailReportFileList($target);
						$this->_addToSuccessList($this->_files[$inputfieldname]["name"][$i], 
												$newname, 
												$this->_files[$inputfieldname]["size"][$i], 
												$this->_files[$inputfieldname]["type"][$i]);
						if (!@chmod($target, UPLOADER_DEFAULT_CHMOD)) {
							$this->_errorTrigger($current_filename, 12);
						}
					} else {
						$this->_errorTrigger($current_filename, 9);
					}
					break;
				default: // will raise an error message because missing overwrite mode
					// will upload anyway, but will abort in case a duplicate already exists
					$this->_errorTrigger($current_filename, 11);
					if ($this->_checkFileExists($this->_filenameClean)) {
						$this->_errorTrigger($current_filename, 5);
					} else {
						if (@move_uploaded_file($this->_files[$inputfieldname]["tmp_name"][$i], $target)) {
							$this->_errorTrigger($current_filename, 0);
							$this->_addToMailReportFileList($target);
							$this->_addToSuccessList($this->_files[$inputfieldname]["name"][$i], 
													$this->_filenameClean, 
													$this->_files[$inputfieldname]["size"][$i], 
													$this->_files[$inputfieldname]["type"][$i]);
							if (!@chmod($target, UPLOADER_DEFAULT_CHMOD)) {
								$this->_errorTrigger($current_filename, 12);
							}
						} else {
							$this->_errorTrigger($current_filename, 9);
						}
					}
					break;
				} // end switch
			
			} // end if
		} // end for
		
		// send out an email-reporting to admin(s), if necessary
		$this->_triggerEmailReport();
		
		return true;
	}
	
	/**
	 *	sets the maximum size in bytes a file can have for upload
	 *	@access public
	 *	@param int $size (size in bytes)
	 *	@return void
	 */
	function setMaxFilesize($size)
	{
		$this->_maxFilesize = (int) $size;
	}
	
	/**
	 *	sets the filetypes that are allowed for upload
	 *	@access public
	 *	@param Array $filetypes (list of allowed filetypes)
	 *	@return void
	 */
	function setAllowedFiletypes($filetypes)
	{
		if (is_array($filetypes)) {
			foreach ($filetypes as $type) {
				array_push($this->_allowedFiletypes, $type);
			}
		}
	}
	
	/**
	 *	prints out a reporting of the upload
	 *	@access public
	 *	@param String $lang (language of the output)
	 *	@return void
	 */
	function printReport($lang="en")
	{
		$no = 1;
		foreach ($this->_errors as $key=>$val) { // $key: filename, $val: array with error/ok messages
			// if we have multiple files, also show the current no of the processed file
			if (sizeof($this->_errors) > 1) {
				$key = "#" . $no . " " . $key;
			}
			
			echo "<br />\n" . $key . ":\n<br />\n";
			
			if (empty($val)) {
				echo "ok.\n<br />\n";
			} else {
				$errs = array();
				$errs = $this->_translateErrors($val, $lang);
				foreach ($errs as $err) {
					echo $err . "<br />\n";
				}
			}
			$no++;
		}
	}
	
	/**
	 *	returns reporting of the upload in an array
	 *	@access public
	 *	@return Array
	 */
	function getReport()
	{
		return $this->_errors;
	}
	
	/**
	 *	returns reporting of the successful uploads with complete infos
	 *	@access public
	 *	@return Array
	 */
	function getReportSuccess()
	{
		return $this->_successList;
	}
	
	/**
	 *	send an email to the recipient_list after upload(s) finished
	 *	@access public
	 *	@param String $sender email address of the sender
	 *	@param String $subject
	 *	@param Array $recipient_list array with all recipients of the email-reporting
	 *	@param String $lang
	 *	@return void
	 */
	function setEmailReport ($sender, $subject, $recipient_list, $lang = "en")
	{
			$this->_mailReportRecipients = $recipient_list;
			$this->_mailReportLang = $lang;
			$this->_mailReportSender = $sender;
			$this->_mailReportSubject = $subject;
	}
	
	/**
	 *	set the value uploaded files are chmodded to
	 *	@access public
	 *	@param int $chmod
	 *	@return void
	 */
	function setChmod($chmod)
	{
		$this->_chmod = $chmod;
	}
	
	
	/******************************************************************/
	/************************* private methods ************************/
	/******************************************************************/
	
	
	/**
	 *	checks uploaded files for errors, also sets name of current file
	 *	@access private
	 *	@param String $inputfieldname
	 *	@param int $i (current index of files array)
	 *	@param string $current_filename (original name of file uploaded)
	 *	@return bool
	 */
	function _checkFileForErrors($inputfieldname, $i, &$current_filename)
	{
		$ok = true; // set the current status of the file to ok
		
		$current_filename = ""; // set up the messaging service:
		if (!empty($this->_files[$inputfieldname]["name"][$i])) {
			$current_filename = $this->_files[$inputfieldname]["name"][$i];
		} else {
			$current_filename = "_no_file_";
		}
		
		$this->_errorInit($current_filename); // je file error array (leer) initialisieren
		
		// 1. check if file was really uploaded
		if (!is_uploaded_file($this->_files[$inputfieldname]["tmp_name"][$i])) {
			$ok = false;
			if ($this->_phpvers >= 420 && $this->_files[$inputfieldname]["error"][$i] == 4 ) {
				$this->_errorTrigger($current_filename, 4);
			} else {
				$this->_errorTrigger($current_filename, 7);
			}
		}
		// 2. check filesize:
		if (!$this->_checkFilesize($this->_files[$inputfieldname]["size"][$i])) {
			$ok = false;
			$this->_errorTrigger($current_filename, 6);
		}
		// 3. check, if file was really uploaded and did not abort
		if ($this->_files[$inputfieldname]["tmp_name"][$i] == "none") {
			$ok = false;
			$this->_errorTrigger($current_filename, 7);
		}
		// 4. check if filetype has a valid extension
		if (!$this->_checkIsFiletypeValid($this->_files[$inputfieldname]["name"][$i]) && $ok == true) {
			$ok = false;
			$this->_errorTrigger($current_filename, 8);
		}
		
		if ($this->_phpvers >= 420) { // error codes available from php4.2.0+
			if ($this->_files[$inputfieldname]["error"][$i] == 1) {
				$ok = false;
				$this->_errorTrigger($current_filename, 1);
			} elseif ($this->_files[$inputfieldname]["error"][$i] == 2) {
				$ok = false;
				$this->_errorTrigger($current_filename, 2);
			} elseif ($this->_files[$inputfieldname]["error"][$i] == 3) {
				$ok = false;
				$this->_errorTrigger($current_filename, 3);
			}
		}
		
		return $ok;
	}
	
	/**
	 *	fills the array $this->_files with values of $_FILES
	 *	@access private
	 *	@param String $inputfieldname
	 *	@return void
	 */
	function _fillFilesArray($inputfieldname)
	{
		$files =& $_FILES;
		
		if (is_array($files[$inputfieldname]["tmp_name"])) { // <input type="file" name="upload[]">
			$this->_files = $files;
		} else {  // <input type="file" name="upload">
			$this->_files[$inputfieldname]["name"][0] = $files[$inputfieldname]["name"];
			$this->_files[$inputfieldname]["tmp_name"][0] = $files[$inputfieldname]["tmp_name"];
			$this->_files[$inputfieldname]["type"][0] = $files[$inputfieldname]["type"];
			$this->_files[$inputfieldname]["size"][0] = $files[$inputfieldname]["size"];
			if ($this->_phpvers >= 420) {
				$this->_files[$inputfieldname]["error"][0] = $files[$inputfieldname]["error"];
			}
		}
	}
	
	/**
	 *	checks, if the file is of a valid type
	 *	@access private
	 *	@param String $file (absolute path to the file)
	 *	@return Bool
	 */
	function _checkIsFiletypeValid($file)
	{
		$path = pathinfo($file);
		if (!in_array(strtolower(@$path["extension"]), $this->_allowedFiletypes)) {
			return false;
		}
		return true;
	}
	
	/**
	 *	checks, if the filesize is ok
	 *	@access private
	 *	@param int $filesize (filesize in bytes of the respective file)
	 *	@return Bool
	 */
	function _checkFilesize($filesize)
	{
		if ($this->_maxFilesize > $filesize) {
			return true;
		}
		return false;
	}
	
	/**
	 *	checks, if the respective file already exists
	 *	@access private
	 *	@param String $file (filename of the uploaded file)
	 *	@return Bool
	 */
	function _checkFileExists($file)
	{
		if (file_exists($this->_abspath . $this->_delimiter . $file)) {
			return true;
		}
		return false;
	}
	
	/**
	 *	automatically renames uploaded file, if the respective file(name) already exists
	 *	@access private
	 *	@param String $filename (filename of the uploaded file)
	 *	@return String $new_filename
	 */
	function _autoRename($filename)
	{
		$filename_full = "";
		$filename_extension = "";
		$filename_noExt = "";
		$path = array();
		
		$path = pathinfo($filename);
		$filename_full = @$path["basename"];
		$filename_extension = @$path["extension"];
		if (!empty($filename_extension)) {
			$filename_extension = "." . $filename_extension;
			$position = strpos($filename_full, $filename_extension); // get the position of ".extension"
			$filename_noExt = substr($filename, 0, $position); // extract the filename without extension
		} else {
			$filename_noExt = $filename_full;
		}
		// dann den neuen namen erstellen
		$n = 0;
		$copy = "";
		while(file_exists($this->_abspath . $this->_delimiter . $filename_noExt . $copy . $filename_extension))
		{
			if ($n<=9) $n = "00" . $n;
			if ($n<=99 && $n>=10) $n = "0" . $n;
			if ($n<=999 && $n>=100) $n = "" . $n;
			$copy = "_" . $n;
			$n++;
		}
		// return new filename
		return $filename_noExt . $copy . $filename_extension;
	}
	
	
	/**
	 *	translates the error codes to a readable string
	 *	@access private
	 *	@param Array $errorCode
	 *	@param String $lang (the language to be displayed)
	 */
	function _translateErrors($errorCode, $lang="en")
	{
		$err = array();
		switch (strtolower($lang)) {
		case "de":
			// standard error messages provided by $_FILES["error"] since 4.2.0
			$err[0] = "Die Datei wurde erfolgreich hochgeladen.";
			$err[1] = "Die hochgeladene Datei &uuml;berschreitet den Wert f&uuml;r upload_max_filesize in php.ini.";
			$err[2] = "Die hochgeladene Datei &uuml;berschreitet den Wert f&uuml;r MAX_FILE_SIZE, der im HTML Formular angegeben wurde.";
			$err[3] = "Die hochgeladene Datei wurde nur teilweise hochgeladen.";
			$err[4] = "Es wurde keine Datei hochgeladen.";
			// other messages
			$err[5] = "Die Datei existiert bereits auf dem Server und wurde nicht &uuml;berschrieben.";
			$err[6] = "Dateigr&ouml;sse &uuml;berschreitet das Limit.";
			$err[7] = "Datei wurde nicht hochgeladen.";
			$err[8] = "Dateityp ist nicht f&uuml;r den upload erlaubt.";
			$err[9] = "Ein unerwarteter Fehler trat auf beim Kopieren der Datei.";
			$err[10] = "Es wurde kein Array &uuml;bergeben. Bitte &uuml;bergeben sie das \$_FILES Array an die jeweilige Methode.";
			$err[11] = "Es wurde ein falscher Overwrite mode &uuml;bergeben. Nutze mode 0 als Standard.";
			$err[12] = "CHMOD der Datei war nicht erfolgreich.";
			$err[13] = "Der angegebene Pfad f&uuml;hrt nicht zu einem Verzeichnis.";
			$err[14] = "Das angegebene Verzeichnis hat keine Schreibrechte.";
			break;
		default:
			// standard error messages provided by $_FILES["error"] since 4.2.0
			$err[0] = "The file uploaded with success.";
			$err[1] = "The uploaded file exceeds the upload_max_filesize directive in php.ini.";
			$err[2] = "The uploaded file exceeds the MAX_FILE_SIZE directive that was specified in the html form.";
			$err[3] = "The uploaded file was only partially uploaded.";
			$err[4] = "No file was uploaded.";
			// other messages
			$err[5] = "File already exists and was not overwritten.  Delete the existing file first.";
			$err[6] = "Filesize exceeds limit.";
			$err[7] = "File was not uploaded.";
			$err[8] = "Filetype is not allowed.";
			$err[9] = "An unexpected error occured while trying to copy the file.";
			$err[10] = "No array was passed. Please make sure to pass the \$_FILES array to the respective method.";
			$err[11] = "A wrong overwrite mode was passed, using 0 as standard mode.";
			$err[12] = "CHMOD of the file was not successful.";
			$err[13] = "The given path to the upload-folder does not lead to a directory.";
			$err[14] = "The given directory has no sufficient write-permission.";
			break;
		}
		
		if (is_array($errorCode)) {
			$arrReturn = array();
			foreach ($errorCode as $error) {
				$arrReturn[] = $err[$error];
			}
			return $arrReturn;
		}
		return false;
	}
	
	/**
	 *	sets the absolute path to the storage folder and checks this folder
	 *	@access private
	 *	@param String $path
	 *	@return bool
	 */
	function _setAbspath($path)
	{
		if (substr($path, (strlen($path)-1)) == $this->_delimiter) {
			$path = substr($path, 0, (strlen($path)-1));
		}
		$this->_abspath = $path;
		
		$current_filename = '_path_does_not_exist_';
		if (!is_dir($this->_abspath)) {
			$this->_errorInit($current_filename);
			$this->_errorTrigger($current_filename, 13);
			return false;
		}
		if (!is_writable($this->_abspath)) {
			$this->_errorInit($current_filename);
			$this->_errorTrigger($current_filename, 14);
			return false;
		}
		return true;
	}
	
	/**
	 *	sets the phpversion of the running system
	 *	@access private
	 *	@return void
	 */
	function _setPhpVersion()
	{
		$this->_phpvers = intval(str_replace(".", "", phpversion()));
	}
	
	/**
	 *	initialize an empty error array for a file ("errors" array is an assoc. array)
	 *	@access private
	 *	@param String $filename
	 *	@return void
	 */
	function _errorInit($filename)
	{
		$this->_errors[$filename] = array();
	}
	
	/**
	 *	trigger a specific error for a selected file (fills assoc. array "errors")
	 *	@access private
	 *	@param String $filename
	 *	@param int $errno
	 *	@return void
	 */
	function _errorTrigger($filename, $errno)
	{
		array_push ($this->_errors[$filename], $errno);
	}
	
	/**
	 *	clean up a string so it can be used as a filename
	 *	@access private
	 *	@param String $str
	 *	@return String $str (cleaned up string)
	 */
	function _stringCleaner($str)
	{
		// replace known chars for beauty.. ;)
		$bad = "?????????????????????????????????????????????????????????????????????????????????? ";
		$good = "AaEeIiOoUuCcAaAaAaEeIiOoOoUuSZszYAAAAACEEEEIIIINOOOOOUUUYaaaaaceeeeiiiinooooouuuyy_";
		$arr = array('?' => 'ae', '?' => 'oe', '?' => 'ue', '?' => 'ss', '&' => 'and');
		$str = strtolower(strtr($str, $bad, $good));
		$str = strtr($str, $arr);
		// replace all other chars to ''
		$str = preg_replace('/[^a-zA-Z0-9._-]/', '', $str);
		return $str;
	}
	
	/**
	 *	add uploaded file to the report list that will be used for email-reporting
	 *	@access private
	 *	@param String $file
	 *	@return void
	 */
	function _addToMailReportFileList($file)
	{
		if (is_string($file)) {
			array_push($this->_mailReportFileList, $file);
		}
	}
	
	/**
	 *	add infos about uploaded file to the success list
	 *	@access private
	 *	@param String $nameOriginal
	 *	@param String $nameRenamed
	 *	@param int $filesize
	 *	@param String $filetype
	 *	@return void
	 */
	function _addToSuccessList($nameOriginal, $nameRenamed, $filesize, $filetype)
	{
		$this->_successList[] = array("name_original" => $nameOriginal, 
									"name_renamed" => $nameRenamed, 
									"filesize" => $filesize, 
									"mime_type" => $filetype);
	}
	
	/**
	 *	create full report and send email-notification to admin(s) 
	 *	mail will be sent, only if method set_emailReport was utilized before 
	 *	and files really have been uploaded. otherwise nothing will be sent.
	 *	@access private
	 *	@return void
	 */
	function _triggerEmailReport()
	{
		if (!empty($this->_mailReportRecipients) && !empty($this->_mailReportFileList)) {
			switch ($this->_mailReportLang) {
			case "de":
				$this->_mailReportMessage = "Es wurden Dateien hochgeladen:\n\n";
				break;
			default:
				$this->_mailReportMessage = "There have been files uploaded:\n\n";
				break;
			}
			
			foreach ($this->_mailReportFileList as $file) {
				$this->_mailReportMessage .= $file . "\n";
			}
			
			$from = "From: " . $this->_mailReportSender;
			$subject = $this->_mailReportSubject;
			$msg = $this->_mailReportMessage;
			$msg .= "\n\n";
			$msg .= 'IP: ' . $_SERVER["REMOTE_ADDR"];
			$msg .= "\n";
			$msg .= 'HOST: ' . gethostbyaddr($_SERVER["REMOTE_ADDR"]);
			
			foreach ($this->_mailReportRecipients as $rec) {
				@mail ($rec, $subject, $msg, $from);
			}
		}
	}
	
	
	/******************************************************************/
	/****** deprecated methods, kept for backwards compatibility ******/
	/******************************************************************/
	
	/**
	 *	DEPRECATED uploads files of $_FILES array
	 *	@access public
	 *	@param String $inputfieldname (name of the input field that gathers the files)
	 *	@param String $uploadFolder (folder that will store the uploaded file(s))
	 *	@param int $mode (overwrite mode if file exists: 0|do nothing, 1|overwrite, 2|autorename)
	 *	@return void
	 */
	function upload_file($inputfieldname, $uploadFolder, $mode=0)
	{
		$this->upload($inputfieldname, $uploadFolder, $mode);
	}
	/**
	 *	DEPRECATED sets the maximum size in bytes a file can have for upload
	 *	@access public
	 *	@param int $size (size in bytes)
	 *	@return void
	 */
	function set_maxFilesize($size)
	{
		$this->setMaxFilesize($size);
	}
	/**
	 *	DEPRECATED sets the filetypes that are allowed for upload
	 *	@access public
	 *	@param Array $filetypes (list of allowed filetypes)
	 *	@return void
	 */
	function set_allowedFiletypes($filetypes)
	{
		$this->setAllowedFiletypes($filetypes);
	}
	/**
	 *	DEPRECATED prints out a reporting of the upload
	 *	@access public
	 *	@param String $lang (language of the output)
	 *	@return void
	 */
	function report2screen($lang="en")
	{
		$this->printReport($lang);
	}
	/**
	 *	DEPRECATED returns reporting of the upload in an array
	 *	@access public
	 *	@return Array
	 */
	function report2array()
	{
		return $this->getReport();
	}
	/**
	 *	DEPRECATED returns reporting of the successful uploads with complete infos; new in 0.2.2
	 *	@access public
	 *	@return Array
	 */
	function reportSuccess2array()
	{
		return $this->getReportSuccess();
	}
	/**
	 *	DEPRECATED send an email to the recipient_list after upload(s) finished
	 *	@access public
	 *	@param String $sender email address of the sender
	 *	@param String $subject
	 *	@param Array $recipient_list array with all recipients of the email-reporting
	 *	@param String $lang
	 *	@return void
	 */
	function set_emailReport ($sender, $subject, $recipient_list, $lang = "en")
	{
		$this->setEmailReport($sender, $subject, $recipient_list, $lang);
	}
}
?>