package DADA::MailingList;

use lib qw(./ ../ ../DADA ../perllib);

use Carp qw(croak carp);

use DADA::Config qw(!:DEFAULT); 

require Exporter;
@ISA    = qw(Exporter);
@EXPORT = qw(Create);

use strict;
use vars qw(@EXPORT);

sub Create {

    my ($args) = @_;

    if ( !$args->{-name} ) {
        croak("You must supply a list name in the '-name' paramater.");
    }

    require DADA::MailingList::Settings;
    require DADA::App::Guts;

    my $ml = DADA::MailingList::Settings->new(
        -List     => $args->{-name},
        -new_list => 1
    );
    DADA::App::Guts::make_all_list_files( -List => $args->{-name} );

    return $ml;

}

sub Remove {

    my ($args) = @_;

    if ( !$args->{-name} ) {
        croak("You must supply a list name in the '-name' paramater.");
    }

    my $list = $args->{-name};
    my $delete_backups
        = $args->{-delete_backups} ? $args->{-delete_backups} : 0;

    require DADA::App::Guts;

    require DADA::MailingList::Settings;
    my $ls = DADA::MailingList::Settings->new( -List => $list );
    my $li = $ls->get;

    require DADA::MailingList::Archives;

    #$DADA::MailingList::Archives::dbi_obj = $dbi_handle;
    my $la = DADA::MailingList::Archives->new( -List => $li );

    require DADA::MailingList::Subscribers;
    my $lh = DADA::MailingList::Subscribers->new( -List => $list );

    if ( $delete_backups == 1 ) {

        $ls->removeAllBackups();
        $la->removeAllBackups(1);
    }

    #mostly for the SQL backends
    $lh->remove_this_listtype('list');
    $lh->remove_this_listtype('blacklist');
    $lh->remove_this_listtype('invitelist');

    # No, this shouldn't be in DADA::App::Guts...
    DADA::App::Guts::delete_email_list( 
        -List => $list, 
        -Type => 'list' 
    );
    DADA::App::Guts::delete_email_list(
        -List => $list,
        -Type => 'black_list'
    );
    DADA::App::Guts::delete_email_list(
        -List => $list,
        -Type => 'invitelist'
    );

    # Nor this...
    DADA::App::Guts::delete_list_info( -List => $list );

    $la->delete_all_archive_entries();

    # Nor this...
    DADA::App::Guts::delete_list_template({ -List => $list });

    require DADA::Logging::Usage;
    my $log = new DADA::Logging::Usage;
    $log->mj_log( $list, 'List Removed',
        "remote_host:$ENV{REMOTE_HOST}, ip_address:$ENV{REMOTE_ADDR}" )
        if $DADA::Config::LOG{list_lives};

    return 1;

}

=pod

=head1 NAME

DADA::MailingList - Creates and Removes Dada Mail Mailing Lists

=head1 VERSION

=head1 SYNOPSIS

 use DADA::MailingList; 
 
 my $list = 'foo'; 
 
 
 # Create!
  my $ls = DADA::MailingList::Create({ -name => 'mylist' }); 
 
 # $ls is now a DADA::MailingList::Settings object.
 
 # Remove!
 DADA::MailingList::Remove({ -name => 'mylist' }); 

=head1 DESCRIPTION

This module basically either creates, or removes a list. 


=head1 SUBROUTINES

=head2 Create

 my $ls = DADA::MailingList::Create({ -name => 'mylist' });

Creates all the necessary files for a Dada Mailing List; B<$list> is the 
list shortname of your mailing list - which itself should be no more than 16
characters and should only include letters/numbers.

It returns a B<DADA::MailingList::Settings> object.

=head2 Remove

 DADA::MailingList::Remove({ -name => 'mylist', -delete_backups => 1}); 

Removes a Mailing List. the B<-name> paramater is required. B<-delete_backups> 
is optional, but when set to 1, will remove any backups of the list settings, 
archives or schedules. 

=head1 AUTHOR

Justin Simoni - http://mojo.skazat.com/contact

=head1 LICENCE AND COPYRIGHT

Copyright (c) 1999-2007 Justin Simoni All rights reserved. 

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, 
Boston, MA  02111-1307, USA.

=cut

1;
